/*
 * Copyright (c) 2008, Noel Lemouel
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of this project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND THE CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR THE CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * %[parameter][flags][width][.precision][length]type
 *
 * limitations:
 *   no floats support
 *   unsigned int has to be >= 16bits
 *   '*' or '*m$' not supported in width field
 *   '*' or '*m$' not supported in precision field
 */

#include <stdarg.h>
#include <sysutil.h>

#define FLAG_POS  0x0001
#define FLAG_NEG  0x0002
#define FLAG_ALT  0x0004
#define FLAG_ZRO  0x0008
#define FLAG_SPC  0x0010

#define LMOD_H    0x0001
#define LMOD_HH   0x0002
#define LMOD_L    0x0004
#define LMOD_LL   0x0008
#define LMOD__L   0x0010
/* #define LMOD_Q    0x0020 = LMOD_LL */
#define LMOD_J    0x0040
#define LMOD_Z    0x0080
#define LMOD_T    0x0100

#define FLD_FLAG  0x0001
#define FLD_WDTH  0x0002
#define FLD_PCSN  0x0004
#define FLD_LMOD  0x0008

static const char* chr_lower = "0123456789abcdef";
static const char* chr_upper = "0123456789ABCDEF";

struct fmt_s{
   /* unsigned */ int fields;
   /* unsigned */ int base;
   /* unsigned */ int uc;

   /* unsigned */ int param;
   /* unsigned */ int flags;
   /* unsigned */ int fldwdth;
   /* unsigned */ int fldpcsn;
   /* unsigned */ int fldmod;
};

static int u_intcvrt(char *str, unsigned long val, struct fmt_s* fmtd)
{
   int i;
   char c;
   char tmp[50];

   int strlgth;
   int rtnlgth;

   char* s = tmp;

   /* convert */

   strlgth = 0;
   if (!val){       /* ? */
      *(s++) = '0';
      strlgth++;
   }
   while(val != 0)
   {
      fmtd->uc ? (*(s++)=chr_upper[val%fmtd->base]) : (*(s++)=chr_lower[val%fmtd->base]);
      val = val / fmtd->base;
      strlgth++;
   }
   s--;

   rtnlgth = strlgth;

   /* pad */

   c = '?';
   if(fmtd->fldwdth)
   {
      /* '+' & ' ' are ignored in unsigned conversion ? */
      /* if(fmtd->flags |= FLAG_SPC) c = ' '; */
      if(fmtd->flags && FLAG_ZRO) c = '0';
   }

   i = fmtd->fldwdth - strlgth;
   if(i > 0){
      rtnlgth += i;
      while(i--)
      *(str++) = c;
   }

   /* copy (reverse) */

   while(strlgth--)
      *(str++) = *(s--);
   *s=0; /* ... */

   /* printf("%08x\n", num); */

   return rtnlgth;
}

int vsprintf(char *str, const char *fmt, va_list args)
{
   int i;
   int brk;
   unsigned int strbase;

   char* s;

   struct fmt_s fmtd;

   strbase = (unsigned int)str;

   while(*fmt){

      if(*fmt == '%')
      {
         fmt++;

         fmtd.fields = 0;
         fmtd.param = 0;
         fmtd.flags = 0;

         if(*(fmt) == '%'){ /* % escape a regular '%' char */
            *(str++) = '%';
            fmt++;
            continue;
         }

         /* parameters POSIX extension not C99 */
         if (*(fmt+2) == '$'){
            fmtd.param = *(fmt+1);
            fmt+=3;
            /* ... ignore them ? */
            continue;
         }

         /* flags */
         brk = 0;
         while(!brk){
            switch(*fmt){
               case '+':
                  fmtd.flags |= FLAG_POS;
                  fmtd.fields |= FLD_FLAG;
                  fmt++;
                  break;
               case '-':
                  fmtd.flags |= FLAG_NEG;
                  fmtd.fields |= FLD_FLAG;
                  fmt++;
                  break;
               case '#':
                  fmtd.flags |= FLAG_ALT;
                  fmtd.fields |= FLD_FLAG;
                  fmt++;
                  break;
               case '0':
                  fmtd.flags |= FLAG_ZRO;
                  fmtd.fields |= FLD_FLAG;
                  fmt++;
                  break;
               case ' ':
                  fmtd.flags |= FLAG_SPC;
                  fmtd.fields |= FLD_FLAG;
                  fmt++;
                  break;
               default:
                  brk++;
                  break;
            }
         }

         /* field width */
         i = 1;      /* 10^0 */
         brk = 0;
         fmtd.fldwdth = 0;
         while(!brk){
            /* can't start with '0' => check that later */
            if ((*fmt >= 0x30) && (*fmt <= 0x39)){
               fmtd.fields |= FLD_WDTH;
               fmtd.fldwdth = (fmtd.fldwdth*10) + (*fmt-0x30);
               fmt++;
            }
            else
               /* insert '*' or '*m$' support here */
               brk++;
         }

         /* precision */
         if(*fmt == '.'){
            i = 1;      /* 10^0 */
            brk = 0;
            fmtd.fldpcsn = 0;
            while(brk){
               /* can't start with '0' => check that later */
               if ((*fmt >= 0x30) || (*fmt <= 0x39)){
                  fmtd.fields |= FLD_WDTH;
                  fmtd.fldwdth += (*fmt - 0x30)*i;
                  i*=10;
                  fmt++;
               }
               else
                  /* insert '*' or '*m$' support here */
                  brk++;
            }
         }

         /* length modifier */
         switch(*fmt){
               case 'h':
                  fmtd.fields |= FLD_LMOD;
                  fmt++;
                  /* hh */
                  if(*fmt == 'h'){
                     fmtd.fldmod |= LMOD_HH;
                     fmt++;
                  }else{
                     fmtd.fldmod |= LMOD_H;
                  }
                  break;
               case 'l':
                  fmtd.fields |= FLD_LMOD;
                  fmt++;
                  /* ll */
                  if(*fmt == 'l'){
                     fmtd.fldmod |= LMOD_LL;
                     fmt++;
                  }else{
                     fmtd.fldmod |= LMOD_L;
                  }
                  break;
               case 'L':
                  fmtd.fldmod |= LMOD__L;
                  fmtd.fields |= FLD_LMOD;
                  fmt++;
                  break;
               case 'q':
                  /* ('quad'. 4.4BSD and Linux libc5 only. Don't use.)
                   * This is a synonym for ll.
                   */
                  fmtd.fldmod |= LMOD_LL;
                  fmtd.fields |= FLD_LMOD;
                  fmt++;
               case 'j':
                  fmtd.fldmod |= LMOD_J;
                  fmtd.fields |= FLD_LMOD;
                  fmt++;
                  break;
               case 'z':
                  fmtd.fldmod |= LMOD_Z;
                  fmtd.fields |= FLD_LMOD;
                  fmt++;
                  break;
               case 't':
                  fmtd.fldmod |= LMOD_T;
                  fmtd.fields |= FLD_LMOD;
                  fmt++;
                  break;
               default:
                  break;
         }

         /* conversion specifier */
         fmtd.uc = 0;
         switch(*fmt){
               case 'c':
                  /* char */
                  /* gcc: `char' is promoted to `int' when passed through `...' */
                  i = va_arg(args, int /* char */);
                  *(str++) = (char)i;
                  fmt++;
                  break;
               case 'd':
               case 'i':
                  /* int: signed decimal */
                  fmtd.base = 10;
                  str += u_intcvrt(str, va_arg(args, unsigned long), &fmtd);
                  fmt++;
                  break;
               case 'o':
                  /* unsigned int: unsigned octal */

                  break;
               case 'u':
                  /* unsigned int: unsigned decimal */

                  break;
               case 'X':
                  fmtd.uc = 1;
               case 'x':
                  /* unsigned int: unsigned decimal */
                  fmtd.base = 16;
                  str += u_intcvrt(str, va_arg(args, unsigned long), &fmtd);
                  fmt++;
                  break;
               case 's':
                  /* character string */
                  s = va_arg(args, char*);
                  while(*s)
                     *(str++) = *(s++);
                  fmt++;
                  break;
               case 'b':
                  /* custom specifier: binary */
                  fmtd.base = 2;
                  str += u_intcvrt(str, va_arg(args, unsigned long), &fmtd);
                  fmt++;
                  break;
               default:
                  *(str++) = '?';
                  break;
         }
      }else{
         *(str++) = *(fmt++);
      }
   }
   *(str++) = '\0';
   /* return the number of characters printed  */
   return((unsigned int)str - strbase);
}

